% 3D FFT Reconstruction For A Planar Sensor Example
%
% This example demonstrates the use of k-Wave for the reconstruction of a
% three-dimensional photoacoustic wave-field recorded over a planar sensor
% array. The sensor data is simulated using kspaceFirstOrder3D and
% reconstructed using kspacePlaneRecon. It builds on the Simulations In
% Three Dimensions and 2D FFT Reconstruction For A Line Sensor examples.
%
% author: Bradley Treeby
% date: 3rd July 2009
% last update: 16th July 2009
%  
% This example is part of the k-Wave Toolbox (http://www.k-wave.org)

clear all;

% =========================================================================
% SIMULATION
% =========================================================================

% change scale to 2 to reproduce the higher resolution figures given in the
% examples
scale = 1;

% create the computational grid
PML_size = 8;       % size of the PML in pixels
Nx = 64*scale;      % number of pixels in the x direction
Ny = 64*scale;      % number of pixels in the y direction
Nz = 32*scale;      % number of pixels in the z direction
dx = 0.2e-3/scale;  % pixel width [m]
dy = 0.2e-3/scale;  % pixel width [m]
dz = 0.2e-3/scale;  % pixel height [m]
kgrid = makeGrid(Nx, dx, Ny, dy, Nz, dz);

% define the properties of the propagation medium
c = 1500;           % [m/s]
rho = 1000;         % [kg/m^3]

% create initial pressure distribution using makeBall
ball_magnitude = 10;
ball_x_pos = 36*scale;    % pixels
ball_y_pos = 36*scale;    % pixels
ball_z_pos = 16*scale;    % pixels
ball_radius = 3*scale;    % pixels
p0 = ball_magnitude*makeBall(Nx, Ny, Nz, ball_x_pos, ball_y_pos, ball_z_pos, ball_radius);

% smooth the initial pressure distribution and restore the magnitude
p0 = smooth(p0, kgrid, true);

% define a binary planar sensor
sensor_mask = zeros(kgrid.Nz, kgrid.Nx, kgrid.Ny);
sensor_mask(1+PML_size, 1+PML_size:end-PML_size, 1+PML_size:end-PML_size) = 1;

% create the time array
[t_array dt] = makeTime(kgrid, c);

% set the input arguements
input_args = {'PMLSize', PML_size, 'Smooth', false};

% run the simulation
sensor_data = kspaceFirstOrder3D(p0, kgrid, c, rho, t_array, sensor_mask, input_args{:});

% reshape sensor data to x, y, t
[Np Nt] = size(sensor_data);
sensor_data_rs = reshape(sensor_data, sqrt(Np), sqrt(Np), Nt);

% reorder to t, x, y
sensor_data_rs = permute(sensor_data_rs, [3 1 2]);

% reconstruct the initial pressure
p_zxy = kspacePlaneRecon(sensor_data_rs, kgrid.dx, kgrid.dy, dt, c, 'PosCond', true);

% =========================================================================
% VISUALISATION
% =========================================================================

% trim the initial pressure distribution to remove the PML
p0 = p0(1+PML_size:end-PML_size, 1+PML_size:end-PML_size, 1+PML_size:end-PML_size);

% define slice locations
x_slice = ball_x_pos - PML_size;
y_slice = ball_y_pos - PML_size;
z_slice = ball_z_pos - PML_size;

% define a k-space grid using the dimensions of p_zxy
[Nz_recon Nx_recon Ny_recon] = size(p_zxy);
kgrid_recon = makeGrid(Nx_recon, kgrid.dx, Ny_recon, kgrid.dy, Nz_recon, dt*c);

% define a k-space grid with the same z-spacing as the trimmed p0
[Nz_p0 Nx_p0 Ny_p0] = size(p0);
kgrid_interp = makeGrid(Nx_p0, kgrid.dx, Ny_p0, kgrid.dy, Nz_p0, kgrid.dz);

% resample the p_zxy to be the same size as p0
p_zxy_rs = interp3(kgrid_recon.x_off, kgrid_recon.z_off, kgrid_recon.y_off, p_zxy, kgrid_interp.x_off, kgrid_interp.z_off, kgrid_interp.y_off);

% plot the initial pressure
figure;
plot_scale = [-10 10];
subplot(2, 2, 1), imagesc(kgrid_interp.x(1,:,1)*1e3, kgrid_interp.z(:,1,1)*1e3, squeeze(p0(:, :, y_slice)), plot_scale);
title('x-z plane');
axis image;
subplot(2, 2, 2), imagesc(kgrid_interp.y(1,1,:)*1e3, kgrid_interp.z(:,1,1)*1e3, squeeze(p0(:, x_slice, :)), plot_scale);
title('y-z plane');
axis image;
xlabel('(All axes in mm)');
subplot(2, 2, 3), imagesc(kgrid_interp.x(1,:,1)*1e3, kgrid_interp.y(1,1,:)*1e3, squeeze(p0(z_slice, :, :)).', plot_scale);
title('x-y plane');
axis image;
colormap(getColorMap);

% plot the reconstructed initial pressure
figure;
subplot(2, 2, 1), imagesc(kgrid_interp.x(1,:,1)*1e3, kgrid_interp.z(:,1,1)*1e3, squeeze(p_zxy_rs(:, :, y_slice)), plot_scale);
title('x-z plane');
axis image;
subplot(2, 2, 2), imagesc(kgrid_interp.y(1,1,:)*1e3, kgrid_interp.z(:,1,1)*1e3, squeeze(p_zxy_rs(:, x_slice, :)), plot_scale);
title('y-z plane');
axis image;
xlabel('(All axes in mm)');
subplot(2, 2, 3), imagesc(kgrid_interp.x(1,:,1)*1e3, kgrid_interp.y(1,1,:)*1e3, squeeze(p_zxy_rs(z_slice, :, :)).', plot_scale);
title('x-y plane');
axis image;
colormap(getColorMap);

% view reconstruction slice by slice
flyThrough(p_zxy_rs);